(function() {
  // Overlay UI for Assist / Voice / Eye features
  if (window.__gm_injected) return; window.__gm_injected = true;

  function createOverlay() {
    const overlay = document.createElement('div');
    overlay.id = 'gm-assist-overlay';
    Object.assign(overlay.style, {
      position: 'fixed',
      right: '12px',
      bottom: '12px',
      zIndex: 2147483647,
      background: 'rgba(0,0,0,0.6)',
      color: '#fff',
      padding: '8px',
      borderRadius: '8px',
      fontFamily: 'sans-serif'
    });

    const assistBtn = document.createElement('button'); assistBtn.innerText = 'Assist';
    const voiceBtn = document.createElement('button'); voiceBtn.innerText = 'Voice';
    const eyeBtn = document.createElement('button'); eyeBtn.innerText = 'Eye';
    const settingsBtn = document.createElement('button'); settingsBtn.innerText = '⚙';
    [assistBtn, voiceBtn, eyeBtn, settingsBtn].forEach(b => { b.style.margin = '4px'; b.style.padding = '6px'; b.style.cursor = 'pointer'; overlay.appendChild(b); });

    assistBtn.addEventListener('click', onAssist);
    voiceBtn.addEventListener('click', onVoiceToggle);
    eyeBtn.addEventListener('click', onEyeToggle);
    settingsBtn.addEventListener('click', onSettings);

    document.body.appendChild(overlay);
  }

  function getSelectedTextOrInput() {
    const sel = window.getSelection().toString().trim();
    if (sel) return sel;
    const el = document.activeElement;
    if (el && (el.tagName === 'TEXTAREA' || el.tagName === 'INPUT' || el.isContentEditable)) {
      return el.value || el.innerText || '';
    }
    return '';
  }

  function insertTextToActive(text) {
    const el = document.activeElement;
    if (!el) return;
    if (el.tagName === 'TEXTAREA' || el.tagName === 'INPUT') {
      el.value = text;
      el.dispatchEvent(new Event('input', { bubbles: true }));
    } else if (el.isContentEditable) {
      el.innerText = text;
      el.dispatchEvent(new InputEvent('input', { bubbles: true }));
    }
  }

  async function onAssist() {
    const text = getSelectedTextOrInput();
    if (!text) { alert('Select a message or focus the input to assist.'); return; }
    try {
      // prefer local settings override for AI endpoint
      const cfg = JSON.parse(localStorage.getItem('gm-settings') || '{}');
      if (cfg.ai_url) {
        // call the Tauri command which uses backend AI_API_URL by default; we can also proxy via fetch
        try {
          const r = await fetch(cfg.ai_url, { method: 'POST', headers: { 'Content-Type': 'application/json' }, body: JSON.stringify({ prompt: text }) });
          if (r.ok) {
            const body = await r.text();
            insertTextToActive(body);
            alert('Suggestion inserted.');
            return;
          }
        } catch (e) { console.warn('local ai fetch failed', e); }
      }
      const res = await window.__TAURI__.invoke('ai_suggest', { prompt: text });
      if (res) {
        insertTextToActive(res);
        alert('Suggestion inserted.');
      }
    } catch (e) { alert('AI assist failed: ' + e); }
  }

  async function onSettings() {
    // Open settings window
    try {
      await window.__TAURI__.invoke('open_settings_window');
    } catch (e) {
      console.error('Failed to open settings window:', e);
      alert('Failed to open settings. Please try again.');
    }
  }

  let recognition = null;
  function onVoiceToggle() {
    if (!('webkitSpeechRecognition' in window) && !('SpeechRecognition' in window)) {
      alert('SpeechRecognition not available in this webview.');
      return;
    }
    const Rec = window.SpeechRecognition || window.webkitSpeechRecognition;
    if (!recognition) {
      recognition = new Rec();
      recognition.lang = 'en-US';
      recognition.continuous = true;
      recognition.interimResults = false;
      recognition.onresult = (e) => {
        const t = Array.from(e.results).map(r => r[0].transcript).join('');
        insertTextToActive(t);
      };
      recognition.onerror = (e) => { console.warn('speech err', e); };
      recognition.start();
      alert('Voice typing started');
    } else {
      recognition.stop(); recognition = null; alert('Voice typing stopped');
    }
  }

  let eyeActive = false;
  function onEyeToggle() {
    if (!eyeActive) {
      // load webgazer from CDN and start
      const s = document.createElement('script');
      s.src = 'https://webgazer.cs.brown.edu/webgazer.js';
      s.onload = () => {
        try {
          webgazer.setGazeListener(function(data, elapsedTime) {
            if (!data) return;
            // rudimentary: if gaze near bottom-right, focus input
            const w = window.innerWidth; const h = window.innerHeight;
            if (data.x > w * 0.6 && data.y > h * 0.6) {
              const input = document.querySelector('div[contenteditable="true"]') || document.querySelector('textarea') || document.querySelector('input');
              if (input) input.focus();
            }
          }).begin();
          alert('Eye-tracking started (best-effort).');
        } catch (e) { alert('webgazer init failed'); }
      };
      document.head.appendChild(s);
      eyeActive = true;
    } else {
      try { webgazer && webgazer.end(); } catch (e) {}
      eyeActive = false; alert('Eye-tracking stopped');
    }
  }

  // initialize
  try { createOverlay(); } catch (e) { console.warn('inject init failed', e); }
})();
